Attribute VB_Name = "EditorModule"
Option Explicit

' ---------------------------------------------------------------
'  This procedure saves an EasyNet diagram in a sequential file.
'  It saves:
' - the version number
' - the nodes count
' - the links count
' - every properties of each node (except Picture property)
' - every properties of each link.
'
'  Picture property is not saved but you may instead manage
' a correspondance between node types and pictures. For
' instance when you load your file, your VB application knows
' that node of type 1 have one icon, nodes of type 2 have another
' icon, etc...
'
'  This program is just an example to show how an EasyNet file
' may be saved to disk.
'  Properties that applied to the whole diagram like FontSize or
' FontName are not saved here.
'  You may proceed differently: for instance, use a binary or
' a random file and save only the properties you need for your
' application.
'  You may consider this program as a starting point to write
' your EasyNet saving/loading procedures adapted to your needs.
'
' THE CODE PROVIDED HEREUNDER IS PROVIDED AS IS WITHOUT WARRANTY
' OF ANY KIND.
' ---------------------------------------------------------------
'

' Following type is used for loading only.

Type ItemRec
  Type As Integer
  Data As Long
  FillColor As Long
  ForeColor As Long
  DrawColor As Long
  DrawWidth As Integer
  DrawStyle As Integer
  Sleeping As Integer
  Hiding As Integer
  ItemTag As String
  Text As String
  Shape As Integer
  Transparent As Integer
  Alignment As Integer
  AutoSize As Integer
  X1 As Long
  Y1 As Long
  X2 As Long
  Y2 As Long
  Oriented As Integer
  LinkHead As Integer
  OwnerNode As Long
  SrcNode As Long
  DstNode As Long
  Points As Integer
  FontName As String
  FontSize As Single
  FontBold As Boolean
  FontItalic As Boolean
  FontUnderline As Boolean
  FontStrikethru As Boolean
End Type

Sub SaveEasyNetFile(Net1 As Control, Filename$)
  Dim i%, j%, length%, PointCount%
  Dim TextLength%, TagLength%
  Dim Text$, ItemTag$, s$, CR$
  Dim nodeId&, Owner&, Org&, Dst&, l&, item&, X1&, Y1&, X2&, Y2&
  Dim NodeCount&, LinkCount&
  Dim node() As Long
  Dim link() As Long
  Dim ptx() As Long
  Dim pty() As Long

  CR = Chr$(13)

  With Net1
    Open Filename For Output As 1
    Print #1, "EASYNET VERSION = " + Format$(.Version)
  
    ' Node count
    NodeCount = .GetNodesCount
    Print #1, "Nodes = " + Format$(NodeCount)

    ' Link count
    LinkCount = .GetLinksCount
    Print #1, "Links = " + Format$(LinkCount)

    ' Allocate array to store node and link identifiers.
    If NodeCount > 0 Then
      ReDim node(1 To NodeCount)
      .GetNodesArray NodeCount, node(1)
    End If
    If LinkCount > 0 Then
      ReDim link(1 To LinkCount)
      .GetLinksArray LinkCount, link(1)
    End If

    '-----------
    ' Save nodes
    '-----------

    ' For each node:
    '   - make it the current one
    '   - save its properties in the file

    For i = 1 To NodeCount
      item = node(i)

      ' Get text and its length
      Text = .GetItemText(item)
      TextLength = Len(Text)

      ' Get tag and its length
      ItemTag = .GetItemTag(item)
      TagLength = Len(ItemTag)

      ' Find owner
      Owner = 0
      nodeId = .GetNodeOwner(item)
      For j = 1 To NodeCount
        If node(j) = nodeId Then
          Owner = j
          Exit For
        End If
      Next

      ' Save current node properties
      Print #1, "Begin Node " + Format$(i)
      Print #1, "  Owner = " + Format$(Owner)
      Print #1, "  Type = " + Str(.GetItemShort(item))
      Print #1, "  Data = " + Str(.GetItemLong(item))
      Print #1, "  ForeColor = " + Str(.GetItemForeColor(item))
      Print #1, "  FillColor = " + Str(.GetNodeFillColor(item))
      Print #1, "  DrawColor = " + Str(.GetItemDrawColor(item))
      Print #1, "  DrawWidth = " + Str(.GetItemDrawWidth(item))
      Print #1, "  DrawStyle = " + Str(.GetItemDrawStyle(item))
      Print #1, "  Alignment = " + Str(.GetNodeAlignment(item))
      Print #1, "  AutoSize = " + Str(.GetNodeAutoSize(item))
      Print #1, "  Shape = " + Str(.GetNodeShape(item))
     
      .GetNodeRect item, X1, Y1, X2, Y2
      Print #1, "  X1 = " + Str(X1)
      Print #1, "  Y1 = " + Str(Y1)
      Print #1, "  X2 = " + Str(X2)
      Print #1, "  Y2 = " + Str(Y2)
      If .IsNodeTransparent(item) = False Then
        Print #1, "  Transparent = " + "0"
      Else
        Print #1, "  Transparent = " + "-1"
      End If
      If .IsItemSleeping(item) = False Then
        Print #1, "  Sleeping = " + "0"
      Else
        Print #1, "  Sleeping = " + "-1"
      End If
      If .IsItemHiding(item) = False Then
        Print #1, "  Hiding = " + "0"
      Else
        Print #1, "  Hiding = " + "-1"
      End If
      
      If TextLength > 0 Then
        s = Text
        length = InStr(s, CR)
        While length > 0
          Print #1, "  Text =" + Left$(s, length - 1)
          s = Mid$(s, length + 2)
          length = InStr(s, CR)
        Wend
        Print #1, "  Text =" + s
      End If
      
      If TagLength > 0 Then
        s = ItemTag
        length = InStr(s, CR)
        While length > 0
          Print #1, "  ItemTag =" + Left$(s, length - 1)
          s = Mid$(s, length + 2)
          length = InStr(s, CR)
        Wend
        Print #1, "  ItemTag =" + s
      End If
      
      SaveFont Net1, item

      Print #1, "End"
    Next i

    '-----------
    ' Save links
    '-----------
  
    ' For each link:
    '   - find its origin and destination nodes
    '   - save its properties in the file

    For i = 1 To LinkCount
      ' Make link the current item
      item = link(i)

      ' Find origin
      Org = 0
      nodeId = .GetLinkOrg(item)
      For j = 1 To NodeCount
        If node(j) = nodeId Then
          Org = j
          Exit For
        End If
      Next

      ' Find destination
      Dst = 0
      nodeId = .GetLinkDst(item)
      For j = 1 To NodeCount
        If node(j) = nodeId Then
          Dst = j
          Exit For
        End If
      Next
    
      ' Get text and its length
      Text = .GetItemText(item)
      TextLength = Len(Text)

      ' Get tag and its length
      ItemTag = .GetItemTag(item)
      TagLength = Len(ItemTag)

      ' Get Number of points
      PointCount = .GetLinkPointCount(item)

      ' Get points
      If PointCount > 0 Then
        ReDim ptx(1 To PointCount)
        ReDim pty(1 To PointCount)
        For l = 1 To PointCount
          ptx(l) = .GetLinkPointX(item, l)
          pty(l) = .GetLinkPointY(item, l)
        Next
      End If

      ' Save current link properties
      Print #1, "Begin Link " + Format$(i)
      Print #1, "  Type = " + Str(.GetItemShort(item))
      Print #1, "  Data = " + Str(.GetItemLong(item))
      Print #1, "  ForeColor = " + Str(.GetItemForeColor(item))
      Print #1, "  DrawColor = " + Str(.GetItemDrawColor(item))
      Print #1, "  DrawWidth = " + Str(.GetItemDrawWidth(item))
      Print #1, "  DrawStyle = " + Str(.GetItemDrawStyle(item))
      Print #1, "  LinkHead = " + Str(.GetLinkArrowHead(item))
      Print #1, "  Src = " + Format$(Org)
      Print #1, "  Dst = " + Format$(Dst)
      If .IsLinkOriented(item) = False Then
        Print #1, "  Oriented = " + "0"
      Else
        Print #1, "  Oriented = " + "-1"
      End If
      If .IsItemSleeping(item) = False Then
        Print #1, "  Sleeping = " + "0"
      Else
        Print #1, "  Sleeping = " + "-1"
      End If
      If .IsItemHiding(item) = False Then
        Print #1, "  Hiding = " + "0"
      Else
        Print #1, "  Hiding = " + "-1"
      End If
    
      Print #1, "  Points = " + Format$(PointCount)
      If PointCount > 0 Then
        For l = 1 To PointCount
          Print #1, "    " + Format$(ptx(l)) + "," + Format$(pty(l))
        Next
      End If
      If TextLength > 0 Then
        s = Text
        length = InStr(s, CR)
        While length > 0
          Print #1, "  Text =" + Left$(s, length - 1)
          s = Mid$(s, length + 2)
          length = InStr(s, CR)
        Wend
        Print #1, "  Text =" + s
      End If
      If TagLength > 0 Then
        s = ItemTag
        length = InStr(s, CR)
        While length > 0
          Print #1, "  ItemTag =" + Left$(s, length - 1)
          s = Mid$(s, length + 2)
          length = InStr(s, CR)
        Wend
        Print #1, "  ItemTag =" + s
      End If
            
      SaveFont Net1, item

      Print #1, "End"
    Next i
    
    Erase node
    Erase link
    Erase ptx
    Erase pty

    ' Close file
    Close
  End With
End Sub


'----------------------------------------------------
' (See comment of SaveEasyNetFile subroutine.)
'----------------------------------------------------

Sub OpenEasyNetFile(Net1 As Control, Filename$)
  Dim s$, value$, keyword$, CRLF$
  Dim length%, i%, NodeCount%, LinkCount%, Version%
  Dim ir As ItemRec
  Dim l&, item&
  Dim ptx() As Long
  Dim pty() As Long
  Dim node() As Long
  Dim Owner() As Integer
  Dim f As New StdFont

  CRLF = Chr$(13) + Chr$(10)
  Open Filename For Input As #1
  
  With Net1
    Line Input #1, s  ' Version
    Version = Val(Mid$(s, InStr(s, "=") + 1))
    If Version <> .Version Then
      MsgBox "File created by another EasyNet version!", 0, "Editor"
      'Exit Sub
    End If

    ' Node count
    Line Input #1, s
    NodeCount = Val(Mid$(s, InStr(s, "=") + 1))

    ' Link count
    Line Input #1, s
    LinkCount = Val(Mid$(s, InStr(s, "=") + 1))

    If NodeCount = 0 Then
      Close
      Exit Sub
    End If
  
    ReDim node(1 To NodeCount)
    ReDim Owner(1 To NodeCount)

    ' Load all nodes
    For i = 1 To NodeCount
      Line Input #1, s  ' Skip Begin keyword
      length = InStr(s, " ")
      keyword = Left$(s, length - 1)
    
      If keyword = "Begin" Then
        ' Default values
        ir.Type = 0
        ir.Data = 0
        ir.ItemTag = ""
        ir.Text = ""
        ir.ForeColor = .GetItemForeColor(0)
        ir.FillColor = .GetNodeFillColor(0)
        ir.DrawColor = .GetItemDrawColor(0)
        ir.DrawWidth = .GetItemDrawWidth(0)
        ir.DrawStyle = .GetItemDrawStyle(0)
        ir.Sleeping = .IsItemSleeping(0)
        ir.Hiding = .IsItemHiding(0)
        ir.Shape = .GetNodeShape(0)
        ir.Alignment = .GetNodeAlignment(0)
        ir.AutoSize = .GetNodeAutoSize(0)
        ir.Transparent = .IsNodeTransparent(0)
        ir.X1 = 0
        ir.Y1 = 0
        ir.X2 = 0
        ir.Y2 = 0
        ir.OwnerNode = 0
        ir.FontName = .FontName
        ir.FontSize = .FontSize
        ir.FontBold = .FontBold
        ir.FontItalic = .FontItalic
        ir.FontUnderline = .FontUnderline
        ir.FontStrikethru = .FontStrikethru

        Do
          Line Input #1, s  ' Skip Begin keyword
          s = LTrim$(s)
          length = InStr(s, " ")
          If length > 0 Then
            keyword = Left$(s, length - 1)
          Else
            keyword = s
          End If
          If keyword = "End" Then
            Exit Do
          End If
          value = Mid$(s, length + 2)

          ' Load each node property
          Select Case keyword
          Case "Type"
            ir.Type = Val(value)
          Case "Data"
            ir.Data = Val(value)
          Case "FillColor"
            ir.FillColor = Val(value)
          Case "ForeColor"
            ir.ForeColor = Val(value)
          Case "DrawColor"
            ir.DrawColor = Val(value)
          Case "DrawWidth"
            ir.DrawWidth = Val(value)
          Case "DrawStyle"
            ir.DrawStyle = Val(value)
          Case "Sleeping"
            ir.Sleeping = Val(value)
          Case "Hiding"
            ir.Hiding = Val(value)
          Case "Transparent"
            ir.Transparent = Val(value)
          Case "Alignment"
            ir.Alignment = Val(value)
          Case "AutoSize"
            ir.AutoSize = Val(value)
          Case "Shape"
            ir.Shape = Val(value)
          Case "X1"
            ir.X1 = Val(value)
          Case "X2"
            ir.X2 = Val(value)
          Case "Y1"
            ir.Y1 = Val(value)
          Case "Y2"
            ir.Y2 = Val(value)
          Case "Owner"
            ir.OwnerNode = Val(value)
          Case "ItemTag"
            If ir.ItemTag = "" Then
              ir.ItemTag = value
            Else
              ir.ItemTag = ir.ItemTag + CRLF + value
            End If
          Case "Text"
            If ir.Text = "" Then
              ir.Text = value
            Else
              ir.Text = ir.Text + CRLF + value
            End If
          Case "FontName"
            ir.FontName = value
          Case "FontSize"
            ir.FontSize = Val(value)
          Case "FontBold"
            ir.FontBold = Val(value)
          Case "FontItalic"
            ir.FontItalic = Val(value)
          Case "FontUnderline"
            ir.FontUnderline = Val(value)
          Case "FontStrikethru"
            ir.FontStrikethru = Val(value)
          End Select
        Loop

        ' Create Node
        item = .AddNodeItem

        ' For each node, store its identifier
        ' (will be used for links loading and for owner nodes)
        node(i) = item
        Owner(i) = ir.OwnerNode
    
        .SetItemShort item, ir.Type
        .SetItemLong item, ir.Data
        .SetNodeFillColor item, ir.FillColor
        .SetItemForeColor item, ir.ForeColor
        .SetItemDrawColor item, ir.DrawColor
        .SetItemDrawWidth item, ir.DrawWidth
        .SetItemDrawStyle item, ir.DrawStyle
        .SetItemHiding item, ir.Hiding
        .SetNodeAlignment item, ir.Alignment
        .SetNodeAutoSize item, ir.AutoSize
        .SetNodeShape item, ir.Shape
        .SetNodeTransparent item, ir.Transparent
        .SetNodeRect item, ir.X1, ir.Y1, ir.X2, ir.Y2
        .SetItemTag item, ir.ItemTag
        .SetItemText item, ir.Text
        
        f.Name = ir.FontName
        f.Size = ir.FontSize
        f.Bold = ir.FontBold
        f.Italic = ir.FontItalic
        f.Underline = ir.FontUnderline
        f.Strikethrough = ir.FontStrikethru
        .SetItemFont item, f

        .SetItemSleeping item, ir.Sleeping
      End If
    Next i

    ' Manage owner nodes
    For i = 1 To NodeCount
      If Owner(i) <> 0 Then
        .SetNodeOwner node(i), node(Owner(i))
      End If
    Next i

    ' List of link
    For i = 1 To LinkCount
      Line Input #1, s  ' Skip Begin keyword
      length = InStr(s, " ")
      keyword = Left$(s, length - 1)
    
      If keyword = "Begin" Then
        ' Default values
        ir.Type = 0
        ir.Data = 0
        ir.ItemTag = ""
        ir.Text = ""
        ir.ForeColor = .GetItemForeColor(0)
        ir.DrawColor = .GetItemDrawColor(0)
        ir.DrawWidth = .GetItemDrawWidth(0)
        ir.DrawStyle = .GetItemDrawStyle(0)
        ir.Sleeping = .IsItemSleeping(0)
        ir.Hiding = .IsItemHiding(0)
        ir.Oriented = .IsLinkOriented(0)
        ir.LinkHead = .GetLinkArrowHead(0)
        ir.SrcNode = 0
        ir.DstNode = 0
        ir.Points = 0
        ir.FontName = .FontName
        ir.FontSize = .FontSize
        ir.FontBold = .FontBold
        ir.FontItalic = .FontItalic
        ir.FontUnderline = .FontUnderline
        ir.FontStrikethru = .FontStrikethru
  
        Do
          Line Input #1, s  ' Skip Begin keyword
          s = LTrim$(s)
          length = InStr(s, " ")
          If length > 0 Then
            keyword = Left$(s, length - 1)
          Else
            keyword = s
          End If
          If keyword = "End" Then
            Exit Do
          End If
          value = Mid$(s, length + 2)

          ' Load each link property
          Select Case keyword
          Case "Type"
            ir.Type = Val(value)
          Case "Data"
            ir.Data = Val(value)
          Case "ForeColor"
            ir.ForeColor = Val(value)
          Case "DrawColor"
            ir.DrawColor = Val(value)
          Case "DrawWidth"
            ir.DrawWidth = Val(value)
          Case "DrawStyle"
            ir.DrawStyle = Val(value)
          Case "Sleeping"
            ir.Sleeping = Val(value)
          Case "Hiding"
            ir.Hiding = Val(value)
          Case "Oriented"
            ir.Oriented = Val(value)
          Case "LinkHead"
            ir.LinkHead = Val(value)
          Case "ItemTag"
            If ir.ItemTag = "" Then
              ir.ItemTag = value
            Else
              ir.ItemTag = ir.ItemTag + CRLF + value
            End If
          Case "Text"
            If ir.Text = "" Then
              ir.Text = value
            Else
              ir.Text = ir.Text + CRLF + value
            End If
          Case "Src"
            ir.SrcNode = node(Val(value))
          Case "Dst"
            ir.DstNode = node(Val(value))
          Case "Points"
            ir.Points = Val(value)

            ' Get points
            If ir.Points > 0 Then
              ReDim ptx(1 To ir.Points)
              ReDim pty(1 To ir.Points)
              For l = 1 To ir.Points
                Line Input #1, s  ' Read point
                s = LTrim$(s)
                length = InStr(s, ",")
                ptx(l) = Val(Left$(s, length - 1))
                pty(l) = Val(Mid$(s, length + 1))
              Next l
            End If
          Case "FontName"
            ir.FontName = value
          Case "FontSize"
            ir.FontSize = Val(value)
          Case "FontBold"
            ir.FontBold = Val(value)
          Case "FontItalic"
            ir.FontItalic = Val(value)
          Case "FontUnderline"
            ir.FontUnderline = Val(value)
          Case "FontStrikethru"
            ir.FontStrikethru = Val(value)
          End Select
        Loop

        ' Create Link
        item = .AddLinkItem(ir.SrcNode, ir.DstNode)

        .SetItemShort item, ir.Type
        .SetItemLong item, ir.Data
        .SetItemForeColor item, ir.ForeColor
        .SetItemDrawColor item, ir.DrawColor
        .SetItemDrawWidth item, ir.DrawWidth
        .SetItemDrawStyle item, ir.DrawStyle
        .SetItemHiding item, ir.Hiding
        .SetLinkOriented item, ir.Oriented
        .SetLinkArrowHead item, ir.LinkHead
        .SetItemTag item, ir.ItemTag
        .SetItemText item, ir.Text
        .SetLinkPointCount item, ir.Points
        For l = 1 To ir.Points
          .SetLinkPointX item, l, ptx(l)
          .SetLinkPointY item, l, pty(l)
        Next
        
        f.Name = ir.FontName
        f.Size = ir.FontSize
        f.Bold = ir.FontBold
        f.Italic = ir.FontItalic
        f.Underline = ir.FontUnderline
        f.Strikethrough = ir.FontStrikethru
        .SetItemFont item, f

        .SetItemSleeping item, ir.Sleeping
      End If
    Next i
  End With
  
  ' Erase dynamic arrays
  Erase ptx
  Erase pty
  Erase node
  Erase Owner

  ' Close file
  Close
End Sub





Sub SaveFont(Net1 As Control, item As Long)
  Dim f As Font
  Set f = Net1.GetItemFont(item)  ' Item Font

  Print #1, "  FontName = " + f.Name
  Print #1, "  FontSize = " + Str(f.Size)
  If f.Bold = False Then
    Print #1, "  FontBold = " + "0"
  Else
    Print #1, "  FontBold = " + "-1"
  End If
  If f.Italic = False Then
    Print #1, "  FontItalic = " + "0"
  Else
    Print #1, "  FontItalic = " + "-1"
  End If
  If f.Underline = False Then
    Print #1, "  FontUnderline = " + "0"
  Else
    Print #1, "  FontUnderline = " + "-1"
  End If
  If f.Strikethrough = False Then
    Print #1, "  FontStrikethru = " + "0"
  Else
    Print #1, "  FontStrikethru = " + "-1"
  End If
End Sub


